import re
import json
import time
from threading import Thread
from indexers import metadata
from windows import open_window, create_window
from scrapers import external, folders
from modules.debrid import resolve_sources, debrid_enabled, debrid_type_enabled, debrid_valid_hosts
from modules import player, kodi_utils, settings
from modules.source_utils import external_sources, internal_sources, internal_folders_import, scraper_names, get_cache_expiry, pack_enable_check, normalize
from modules.utils import string_to_float, safe_string, remove_accents, get_datetime, adjust_premiered_date
#from modules.kodi_utils import logger

POVPlayer, Manager = player.POVPlayer, external.Manager
progressDialogBG, notification = kodi_utils.progressDialogBG, kodi_utils.notification
show_busy_dialog, hide_busy_dialog, close_all_dialog = kodi_utils.show_busy_dialog, kodi_utils.hide_busy_dialog, kodi_utils.close_all_dialog
get_property, set_property, clear_property = kodi_utils.get_property, kodi_utils.set_property, kodi_utils.clear_property
ls, monitor, sleep, get_setting = kodi_utils.local_string, kodi_utils.monitor, kodi_utils.sleep, kodi_utils.get_setting
check_prescrape_sources, quality_filter, sort_to_top = settings.check_prescrape_sources, settings.quality_filter, settings.sort_to_top
results_xml_style, results_xml_window_number = settings.results_xml_style, settings.results_xml_window_number
default_internal_scrapers, cloud_scrapers = settings.default_internal_scrapers, settings.cloud_scrapers
folder_scrapers = ('folder1', 'folder2', 'folder3', 'folder4', 'folder5')
quality_ranks = {'4K': 1, '1080p': 2, '720p': 3, 'SD': 4, 'SCR': 5, 'CAM': 5, 'TELE': 5}
av1_filter_key, hevc_filter_key, hdr_filter_key, dolby_vision_filter_key = '[B]AV1[/B]', '[B]HEVC[/B]', '[B]HDR[/B]', '[B]D/VISION[/B]'
dialog_format, format_line = '[COLOR %s][B]%s[/B][/COLOR] 4K: %s | 1080p: %s | 720p: %s | SD: %s | Total: %s', '%s[CR]%s[CR]%s'
remaining_format, season_str, show_str, nextep_str, nores_str = ls(32676), ls(32537), ls(32089), ls(32801), ls(32760)

class SourceSelect:
	def __init__(self):
		self.params = {}
		self.clear_properties, self.filters_ignored, self.active_folders = True, False, False
		self.progress_dialog, self.pov_background_url = None, None
		self.threads, self.providers, self.sources, self.internal_scraper_names = [], [], [], []
		self.prescrape_scrapers, self.prescrape_threads, self.prescrape_sources = [], [], []
		self.remove_scrapers = ['external']# needs to be mutable so leave as list.
		self.exclude_list = ['easynews', 'library']# needs to be mutable so leave as list.
		self.internal_resolutions = dict.fromkeys('4K 1080p 720p SD total'.split(), 0)
		self.prescrape, self.disabled_ignored = 'true', 'false'
		self.language = settings.get_language()

	def playback_prep(self, params=None):
		if self.clear_properties: self._clear_properties()
		if params: self.params = params
		params_get = self.params.get
		self.prescrape = params_get('prescrape', self.prescrape) == 'true'
		self.background = params_get('background', 'false') == 'true'
		if self.background: hide_busy_dialog()
		else: show_busy_dialog()
		if 'autoplay' in self.params: self.autoplay = params_get('autoplay', 'false') == 'true'
		else: self.autoplay = settings.auto_play(params_get('media_type'))
		self.disabled_ignored = params_get('disabled_ignored', self.disabled_ignored) == 'true'
		self.ignore_scrape_filters = params_get('ignore_scrape_filters', 'false') == 'true'
		self.media_type = params_get('media_type')
		self.tmdb_id = params_get('tmdb_id')
		self.season = int(params_get('season')) if 'season' in self.params else ''
		self.episode = int(params_get('episode')) if 'episode' in self.params else ''
		self.custom_title = params_get('custom_title')
		self.custom_year = params_get('custom_year')
		self.custom_season = int(params_get('custom_season')) if 'custom_season' in self.params else None
		self.custom_episode = int(params_get('custom_episode')) if 'custom_episode' in self.params else None
		self.active_internal_scrapers = settings.active_internal_scrapers()
		self.active_external = 'external' in self.active_internal_scrapers
		self.display_uncached_torrents = settings.display_uncached_torrents()
		self.ignore_results_filter = settings.ignore_results_filter()
		self.provider_sort_ranks = settings.provider_sort_ranks()
		self.scraper_settings = settings.scraping_settings()
		self.sort_function = settings.results_sort_order()
		self.filter_av1 = settings.filter_status('av1')
		self.filter_hevc = settings.filter_status('hevc')
		self.filter_hdr = settings.filter_status('hdr')
		self.filter_dv = settings.filter_status('dv')
		self.hybrid_allowed = self.filter_hdr in (0, 2)
		self.include_prerelease_results, self.include_3D_results = settings.include_prerelease_3d_results()
		self.quality_filter = self._quality_filter()
		self.full_screen = get_setting('load_action') == '1'
		self.size_filter = int(get_setting('results.size_filter', '0'))
		self.include_unknown_size = get_setting('results.include.unknown.size') == 'true'
		self.sleep_time = settings.display_sleep_time()
		self.timeout = int(get_setting('scrapers.timeout.1', '10')) * 2 if self.disabled_ignored else int(get_setting('scrapers.timeout.1', '10'))
		if get_setting('results.language_filter') == 'true': self.priority_language = get_setting('results.language')
		else: self.priority_language = None
		if not hasattr(self, 'meta'): self.meta = SourceMeta().parse(self.params)
		self.get_sources()

	def get_sources(self):
		results = []
		start_time = time.monotonic()
		self.prepare_internal_scrapers()
		if any(x in self.active_internal_scrapers for x in default_internal_scrapers) and self.prescrape:
			results = self.collect_prescrape_results()
			if results: results = self.process_results(results)
		if not results:
			self.prescrape = False
			if self.active_external: self.activate_external_providers()
			self.orig_results = self.collect_results()
			results = self.process_results(self.orig_results)
		self.meta.update({'scrape_sources': len(results), 'scrape_time': time.monotonic() - start_time})
		if not results: return self._process_post_results()
		self.play_source(results)

	def collect_results(self):
		self.sources.extend(self.prescrape_sources)
		if self.active_folders: self.append_folder_scrapers(self.providers)
		self.providers.extend(internal_sources(self.active_internal_scrapers, self.media_type))
		if self.providers:
			threads = (Thread(target=self.activate_providers, args=(i[0], i[1], False), name=i[2]) for i in self.providers)
			self.threads.extend(threads)
			for i in self.threads: i.start()
		if self.active_external or self.background:
			if self.active_external:
				self.meta.update({'full_screen': self.full_screen, 'scrape_timeout': self.timeout})
				self.external_args = (
					self.meta,
					self.external_providers,
					self.debrid_torrent_enabled,
					self.debrid_hoster_enabled,
					self.internal_scraper_names,
					self.prescrape_sources,
					self.display_uncached_torrents,
					self.progress_dialog,
					self.disabled_ignored
				)
				self.activate_providers('external', Manager, False)
			if self.providers: [i.join() for i in self.threads]
		else: self.scrapers_dialog('internal')
		self._kill_progress_dialog()
		return self.sources

	def collect_prescrape_results(self):
		if self.active_folders:
			if self.autoplay or check_prescrape_sources('folders', self.media_type):
				self.append_folder_scrapers(self.prescrape_scrapers)
				self.remove_scrapers.append('folders')
		self.prescrape_scrapers.extend(internal_sources(self.active_internal_scrapers, self.media_type, True))
		if not self.prescrape_scrapers: return []
		threads = (Thread(target=self.activate_providers, args=(i[0], i[1], True), name=i[2]) for i in self.prescrape_scrapers)
		self.prescrape_threads.extend(threads)
		for i in self.prescrape_threads: i.start()
		self.remove_scrapers.extend(i[2] for i in self.prescrape_scrapers)
		if self.background: [i.join() for i in self.prescrape_threads]
		else: self.scrapers_dialog('pre_scrape')
		self._kill_progress_dialog()
		return self.prescrape_sources

	def process_results(self, results):
		if self.prescrape: self.all_scrapers = self.active_internal_scrapers
		else: self.all_scrapers = list(set(self.active_internal_scrapers + self.remove_scrapers))
		if self.ignore_scrape_filters:
			self.filters_ignored = True
			results = self.sort_results(results)
			results = self._sort_first(results)
		else:
			results = self.filter_results(results)
			results = self.sort_results(results)
			results = self._special_filter(results, hevc_filter_key, self.filter_hevc)
			results = self._special_filter(results, hdr_filter_key, self.filter_hdr)
			results = self._special_filter(results, dolby_vision_filter_key, self.filter_dv)
			results = self._special_filter(results, av1_filter_key, self.filter_av1)
			results = self._sort_first(results)
		return results

	def prepare_internal_scrapers(self):
		if self.active_external and len(self.active_internal_scrapers) == 1: return
		active_internal_scrapers = [i for i in self.active_internal_scrapers if not i in self.remove_scrapers]
		self.active_folders = 'folders' in active_internal_scrapers
		if self.active_folders:
			self.folder_info = self.get_folderscraper_info()
			self.internal_scraper_names = [i for i in active_internal_scrapers if not i == 'folders']
			self.internal_scraper_names += [i[0] for i in self.folder_info]
			self.active_internal_scrapers = active_internal_scrapers
		else:
			self.folder_info = []
			self.internal_scraper_names = active_internal_scrapers[:]
			self.active_internal_scrapers = active_internal_scrapers

	def activate_providers(self, module_type, function, prescrape):
		if module_type == 'external': module = function(*self.external_args)
		elif module_type == 'folders': module = function[0](*function[1])
		else: module = function()
		sources = module.results(self.meta['search_info'])
		if not sources: return
		if prescrape: self.prescrape_sources.extend(sources)
		else: self.sources.extend(sources)

	def activate_external_providers(self):
		self.debrid_enabled = debrid_enabled()
		self.debrid_torrent_enabled = debrid_type_enabled('torrent', self.debrid_enabled)
		self.debrid_hoster_enabled = debrid_valid_hosts(debrid_type_enabled('hoster', self.debrid_enabled))
		if not self.debrid_torrent_enabled and not self.debrid_hoster_enabled:
			self._kill_progress_dialog()
			if ''.join(self.active_internal_scrapers) == 'external': notification(32854)
			self.active_external = False
		else:
#			if not self.debrid_torrent_enabled: self.exclude_list.extend(scraper_names('torrents'))
#			elif not self.debrid_hoster_enabled: self.exclude_list.extend(scraper_names('hosters'))
			external_providers = external_sources(ret_all=self.disabled_ignored)
			self.external_providers = [
				i for i in external_providers if not i[0] in self.exclude_list
				and (i[1].hasEpisodes if self.media_type == 'episode' else i[1].hasMovies)
			]
			if not self.season: return
			self.external_providers = [(i[0], i[1], '') for i in self.external_providers]
			season_packs, show_packs = pack_enable_check(self.meta, self.season, self.episode)
			if not season_packs: return
			pack_capable = [i for i in self.external_providers if i[1].pack_capable]
			if pack_capable:
				self.external_providers.extend([(i[0], i[1], season_str) for i in pack_capable])
			if pack_capable and show_packs:
				self.external_providers.extend([(i[0], i[1], show_str) for i in pack_capable])

	def append_folder_scrapers(self, current_list):
		current_list.extend(internal_folders_import(self.folder_info))

	def get_folderscraper_info(self):
		folder_info = [(get_setting('%s.display_name' % i), i) for i in folder_scrapers]
		return [i for i in folder_info if not i[0] in (None, 'None', '')]

	def scrapers_dialog(self, scrape_type):
		if scrape_type == 'internal':
			scraper_list, _threads, line1_inst, line2_inst = self.providers, self.threads, ls(32096), 'Int:'
		else:
			scraper_list, _threads = self.prescrape_scrapers, self.prescrape_threads,
			line1_inst, line2_inst = '%s %s' % (ls(32829), ls(32830)), 'Pre:'
		self.internal_scrapers = self._get_active_scraper_names(scraper_list)
		if not self.internal_scrapers: return
		int_dialog_hl = get_setting('int_dialog_highlight') or 'dodgerblue'
		line1 = '[COLOR %s][B]%s[/B][/COLOR]' % (int_dialog_hl, line1_inst)
		total_format = '[COLOR %s][B]%s[/B][/COLOR]' % (int_dialog_hl, '%s')
		timeout = self.timeout
		start_time = time.monotonic()
		end_time = start_time + timeout
		if not self.progress_dialog: self._make_progress_dialog()
		while alive_threads := [x.name for x in _threads if x.is_alive() is True]:
			if monitor.abortRequested() or time.monotonic() > end_time: break
			try:
				self._process_internal_results()
				int_totals = [total_format % v for v in self.internal_resolutions.values()]
				current_progress = time.monotonic() - start_time
				line2 = dialog_format % (int_dialog_hl, line2_inst, *int_totals)
				line3 = remaining_format % ', '.join(alive_threads).upper()
				percent = int((current_progress/float(timeout))*100)
				self.progress_dialog.update(format_line % (line1, line2, line3), percent)
			except: pass
			sleep(self.sleep_time)

	def _get_active_scraper_names(self, scraper_list):
		return [i[2] for i in scraper_list]

	def _process_post_results(self):
		if self.ignore_results_filter and self.orig_results: return self._process_ignore_filters()
		return self._no_results()

	def _process_ignore_filters(self):
		if self.autoplay: notification(32686)
		self.autoplay = False
		self.filters_ignored = True
		results = self.sort_results(self.orig_results)
		results = self._sort_first(results)
		return self.play_source(results)

	def _no_results(self):
		hide_busy_dialog()
		if self.background: return notification('%s %s' % (nextep_str, nores_str), 5000)
		notification(32760)

	def _process_internal_results(self):
		for i in self.internal_scrapers:
			win_property = get_property('%s.internal_results' % i)
			if win_property in ('checked', '', None): continue
			try: sources = json.loads(win_property)
			except: continue
			set_property('%s.internal_results' % i, 'checked')
			for k in self.internal_resolutions: self.internal_resolutions[k] += sources.get(k, 0)

	def _quality_filter(self):
		setting = 'results_quality_%s' % self.media_type if not self.autoplay else 'autoplay_quality_%s' % self.media_type
		filter_list = quality_filter(setting)
		if self.include_prerelease_results and 'SD' in filter_list: filter_list += ['SCR', 'CAM', 'TELE']
		return filter_list

	def _clear_properties(self):
		for item in default_internal_scrapers: clear_property('%s.internal_results' % item)
		for item in self.get_folderscraper_info(): clear_property('%s.internal_results' % item[0])

	def _make_progress_dialog(self):
		self.progress_dialog = create_window(('windows.sources', 'ProgressMedia'), 'progress_media.xml', meta=self.meta)
		Thread(target=self.progress_dialog.run).start()

	def _kill_progress_dialog(self):
		try: self.progress_dialog.close()
		except: close_all_dialog()
		try: del self.progress_dialog
		except: pass
		self.progress_dialog = None

	def autoplay_season_premiere(self):
		if self.background or any(x in self.meta for x in ('random', 'random_continual')): return
		if not get_setting('autoplay_exclude_premieres') == 'true': return
		try: unaired_episodes = sum(
			adjust_premiered_date(i['premiered'], self.meta.adjust_hours)[0] > self.meta.current_date
			for i in self.meta.episodes_data
		)
		except: unaired_episodes = True
		if not unaired_episodes: self.autoplay = False

	def display_results(self, results):
		window_style = results_xml_style()
		chosen_item = open_window(
			('windows.sources', 'SourceResults'),
			'sources_results.xml',
			window_style=window_style,
			window_id=results_xml_window_number(window_style),
			results=results,
			meta=self.meta,
			scraper_settings=self.scraper_settings,
			prescrape=self.prescrape,
			filters_ignored=self.filters_ignored
		)
		if not chosen_item: return self._kill_progress_dialog()
		action, chosen_item = chosen_item
		if action == 'play':
			self._kill_progress_dialog()
			return self.play_file(results, chosen_item)
		if action == 'perform_full_search' and self.prescrape:
			self.prescrape, self.clear_properties = False, False
			return self.playback_prep()

	def play_source(self, results):
		if self.media_type == 'episode' and self.autoplay and int(self.episode) == 1: self.autoplay_season_premiere()
		if self.background: self.pov_background_url = self.play_file(results, autoplay=True, background=True)
		elif self.autoplay: return self.play_file(results, autoplay=True)
		else: return self.display_results(results)

	def play_file(self, results, source=None, autoplay=False, background=False):
		def _process():
			for count, item in enumerate(items, 1):
				if not background:
					try:
						if monitor.abortRequested(): break
						elif self.progress_dialog and self.progress_dialog.iscanceled(): break
						percent = int(((total_items := len(items))-count)/total_items*100)
						name = item['name'].replace('.', ' ').replace('-', ' ').upper()
						line1 = item.get('scrape_provider'), item.get('cache_provider'), item.get('provider')
						line1 = ' | '.join(i for i in line1 if i and i != 'external').upper()
						line2 = ' | '.join(i for i in (item.get('size_label', ''), item.get('extraInfo', '')) if i)
						if self.progress_dialog: self.progress_dialog.update(format_line % (line1, line2, name), percent)
						else: progressDialogBG.update(percent, name)
					except: pass
				link = resolve_sources(item, self.meta)
				if not link is None: yield link
		try:
			self._kill_progress_dialog()
			if autoplay:
				items = [i for i in results if not 'Uncached' in i.get('cache_provider', '')]
				if self.filters_ignored: notification(32686)
			else:
				source_index = results.index(source) if source in results else -1
				items = [i for i in results[source_index + 1:] if not 'Uncached' in i.get('cache_provider', '')][:40]
				items.insert(0, source)
			if background: return True if items else None
			if self.full_screen:
				self._make_progress_dialog()
				progress_media = self._kill_progress_dialog
			else:
				progressDialogBG.create('POV', 'POV loading...')
				progress_media = None
			url = next(_process(), None)
			if not self.full_screen: progressDialogBG.close()
			if not url: self._kill_progress_dialog()
			return POVPlayer().run(url, self.meta, progress_media)
		except: pass

	def filter_results(self, results):
		results = [i for i in results if i['quality'] in self.quality_filter]
		if not self.include_3D_results: results = [i for i in results if not '3D' in i['extraInfo']]
		if not self.size_filter: return results
		if self.size_filter == 1:
			duration = self.meta['duration'] or (2400 if self.media_type == 'episode' else 5400)
			max_size = ((0.125 * (0.90 * string_to_float(get_setting('results.size.speed', '20'), '20'))) * duration)/1000
		if self.size_filter == 2:
			max_size = string_to_float(get_setting('results.size.file', '10000'), '10000') / 1000
		if self.include_unknown_size: results = [i for i in results if i['scrape_provider'].startswith('folder') or i['size'] <= max_size]
		else: results = [i for i in results if i['scrape_provider'].startswith('folder') or 0.01 < i['size'] <= max_size]
		return results

	def sort_results(self, results):
		for item in results:
			provider, quality = item['scrape_provider'], item.get('quality', 'SD')
			account_type = item['debrid'].lower() if provider == 'external' else provider.lower()
			item['provider_rank'] = self._get_provider_rank(account_type)
			item['quality_rank'] = self._get_quality_rank(quality)
		results.sort(key=self.sort_function)
		if self.priority_language: results = self._sort_language_to_top(results)
		results = self._sort_uncached_torrents(results)
		clear_property('fs_filterless_search')
		return results

	def _get_provider_rank(self, account_type):
		return self.provider_sort_ranks[account_type] or 11

	def _get_quality_rank(self, quality):
		return quality_ranks[quality]

	def _sort_language_to_top(self, results):
		from xbmc import convertLanguage as cl, ISO_639_1, ISO_639_2
		try:
			language = self.priority_language, cl(self.priority_language, ISO_639_2), cl(self.priority_language, ISO_639_1)
			if self.priority_language == 'Spanish': language += 'latino', 'lat', 'esp'
			pattern = r'\b(%s)\b' % '|'.join(i for i in language if i)
			sort_first = [i for i in results if re.search(pattern, i.get('name_info', ''), re.I)]
			sort_last = [i for i in results if not i in sort_first]
			results = sort_first + sort_last
		except: pass
		return results

	def _sort_uncached_torrents(self, results):
		results.sort(key=lambda k: 'Unchecked' in k.get('cache_provider', ''), reverse=False)
		if self.display_uncached_torrents or get_property('fs_filterless_search') == 'true':
			results.sort(key=lambda k: 'Uncached' in k.get('cache_provider', ''), reverse=False)
			return results
#		uncached = [i for i in results if 'Uncached' in i.get('cache_provider', '')]
#		cached = [i for i in results if not i in uncached]
#		return cached + uncached
		return [i for i in results if not 'Uncached' in i.get('cache_provider', '')]

	def _special_filter(self, results, key, enable_setting):
		if enable_setting == 1:
			if key == dolby_vision_filter_key and self.hybrid_allowed:
				results = [i for i in results if all(x in i['extraInfo'] for x in (key, hdr_filter_key)) or not key in i['extraInfo']]
			else: results = [i for i in results if not key in i['extraInfo']]
		elif enable_setting == 2 and self.autoplay:
			priority_list = [i for i in results if key in i['extraInfo']]
			remainder_list = [i for i in results if not i in priority_list]
			results = priority_list + remainder_list
		elif enable_setting == 3:
			priority_list = lambda k: key in k['extraInfo'] and not 'Uncached' in k.get('cache_provider', '')
			results.sort(key=priority_list, reverse=True)
		return results

	def _sort_first(self, results):
		try:
			sort_first_scrapers = []
			if 'folders' in self.all_scrapers and sort_to_top('folders'): sort_first_scrapers.append('folders')
			sort_first_scrapers.extend([i for i in self.all_scrapers if i in cloud_scrapers and sort_to_top(i)])
			if not sort_first_scrapers: return results
			sort_first = [i for i in results if i['scrape_provider'] in sort_first_scrapers]
			sort_first.sort(key=lambda k: (self._sort_folder_to_top(k['scrape_provider']), k['quality_rank']))
			sort_last = [i for i in results if not i in sort_first]
			results = sort_first + sort_last
		except: pass
		return results

	def _sort_folder_to_top(self, provider):
		if provider == 'folders': return 0
		else: return 1

	nextep_params = []

	@classmethod
	def factory(cls, params):
		if params.get('media_type') == 'episode':
			cls.nextep_callback(params)
			while cls.nextep_params:
				try: cls().playback_prep(cls.nextep_params.pop())
				except: pass
		else: cls().playback_prep(params)

	@classmethod
	def nextep_callback(cls, params):
		if not isinstance(params, dict): return
		cls.nextep_params.insert(0, params)

	@classmethod
	def background_prep(cls, params):
		self = cls()
		self.playback_prep({**params, 'background': 'true'})
		return self.pov_background_url

class SourceMeta(dict):
	def parse(self, params):
		self.params = params
		params_get = self.params.get
		self.background = params_get('background', 'false') == 'true'
		self.media_type = params_get('media_type')
		self.tmdb_id = params_get('tmdb_id')
		self.season = int(params_get('season')) if 'season' in self.params else ''
		self.episode = int(params_get('episode')) if 'episode' in self.params else ''
		self.custom_title = params_get('custom_title')
		self.custom_year = params_get('custom_year')
		self.custom_season = int(params_get('custom_season')) if 'custom_season' in self.params else None
		self.custom_episode = int(params_get('custom_episode')) if 'custom_episode' in self.params else None
		self.update(self.get_meta())
		return self

	def get_meta(self):
		meta = json.loads(self.params['meta']) if 'meta' in self.params else self._get_meta()
		meta.update({
			'background': self.background, 'media_type': self.media_type,
			'season': self.season, 'episode': self.episode
		})
		if self.custom_title: meta['custom_title'] = self.custom_title
		if self.custom_year: meta['custom_year'] = self.custom_year
		if self.custom_season: meta['custom_season'] = self.custom_season
		if self.custom_episode: meta['custom_episode'] = self.custom_episode
		expiry_times = get_cache_expiry(self.media_type, meta, self.season)
		title = metadata.get_title(meta)
		aliases = self._make_alias_dict(title, meta)
		year = self._get_search_year(meta)
		ep_name = self._get_ep_name(meta)
		meta['search_info'] = {
			'media_type': self.media_type, 'expiry_times': expiry_times, 'tmdb_id': self.tmdb_id,
			'imdb_id': meta.get('imdb_id'), 'tvdb_id': meta.get('tvdb_id'), 'year': year, 'aliases': aliases,
			'title': title, 'ep_name': ep_name, 'total_seasons': meta.get('total_seasons', ''),
			'season': self.custom_season or self.season, 'episode': self.custom_episode or self.episode
		}
		return meta

	def _get_meta(self):
		meta_user_info, adjust_hours, current_date = settings.metadata_user_info(), settings.date_offset(), get_datetime()
		if self.media_type == 'episode':
			meta = metadata.tvshow_meta('tmdb_id', self.tmdb_id, meta_user_info, current_date)
			try:
				self.adjust_hours, self.current_date = adjust_hours, current_date
				self.episodes_data = metadata.season_episodes_meta(self.season, meta, meta_user_info)
				ep_data = next((i for i in self.episodes_data if i['episode'] == int(self.episode)))
				meta.update({
					'media_type': 'episode', 'season': ep_data['season'], 'episode': ep_data['episode'],
					'premiered': ep_data['premiered'], 'ep_name': ep_data['title'], 'plot': ep_data['plot']
				})
			except: pass
		else: meta = metadata.movie_meta('tmdb_id', self.tmdb_id, meta_user_info, current_date)
		return meta

	def _make_alias_dict(self, title, meta):
		aliases = []
		meta_title = meta['title']
		original_title = meta['original_title']
		alternative_titles = meta.get('alternative_titles', [])
		country_codes = set([i.replace('GB', 'UK') for i in meta.get('country_codes', [])])
		if meta_title not in alternative_titles: alternative_titles.append(meta_title)
		if original_title not in alternative_titles: alternative_titles.append(original_title)
		if alternative_titles: aliases = [{'title': i, 'country': ''} for i in alternative_titles]
		if country_codes: aliases.extend([{'title': '%s %s' % (title, i), 'country': ''} for i in country_codes])
		normalized = ({'title': normalize(i['title']), 'country': i['country']} for i in aliases)
		aliases.extend(i for i in normalized if not i in aliases)
		return aliases

	def _get_search_year(self, meta):
		if 'custom_year' in meta: return meta['custom_year']
		year = meta.get('year') or '0'
#		if self.active_external and get_setting('search.enable.yearcheck', 'false') == 'true':
		if get_setting('search.enable.yearcheck', 'false') == 'true':
			from indexers.imdb_api import imdb_movie_year
			try: year = str(imdb_movie_year(meta.get('imdb_id')) or year)
			except: pass
		return year

	def _get_ep_name(self, meta):
		if self.media_type == 'episode':
			ep_name = meta.get('ep_name')
			try: ep_name = safe_string(remove_accents(ep_name))
			except: ep_name = safe_string(ep_name)
		else: ep_name = None
		return ep_name

